<?php
/**
 * Frontend class
 *
 * @author  Your Inspiration Themes
 * @package YITH WooCommerce Ajax Navigation
 * @version 1.3.2
 */

if ( ! defined( 'YITH_WCAN' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCAN_Frontend_Premium' ) ) {
	/**
	 * Frontend class.
	 * The class manage all the frontend behaviors.
	 *
	 * @since 1.0.0
	 */
	class YITH_WCAN_Frontend_Premium extends YITH_WCAN_Frontend {

		/**
		 * Constructor method
		 *
		 * @return void
		 */
		public function __construct() {
			parent::__construct();

			// Frontend methods.
			add_filter( 'yith_wcan_body_class', array( $this, 'premium_body_class' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'add_dropdown_styles' ), 20 );

			// Template methods.
			add_action( 'init', array( $this, 'add_active_filters_list' ) );
			add_action( 'init', array( $this, 'add_mobile_modal_opener' ) );

			add_action( 'yith_wcan_before_preset_filters', array( $this, 'filters_title' ), 10, 1 );
			add_action( 'yith_wcan_after_preset_filters', array( $this, 'apply_filters_button' ), 10, 1 );
		}

		/* === FRONTEND METHODS === */

		/**
		 * Enqueue Script for Premium version
		 *
		 * @return void
		 * @author Andrea Grillo <andrea.grillo@yithemes.com
		 * @since  2.0
		 */
		public function enqueue_styles_scripts() {
			parent::enqueue_styles_scripts();

			if ( yith_wcan_can_be_displayed() ) {
				$suffix     = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
				$loader_url = YITH_WCAN_URL . 'assets/images/ajax-loader.gif';

				$options = array(
					'ajax_wc_price_filter'                  => yith_wcan_get_option( 'yith_wcan_enable_ajax_price_filter' ),
					'wc_price_filter_slider'                => yith_wcan_get_option( 'yith_wcan_enable_ajax_price_filter_slider' ),
					'wc_price_filter_slider_in_ajax'        => yith_wcan_get_option( 'yith_wcan_enable_slider_in_ajax' ),
					'wc_price_filter_dropdown'              => yith_wcan_get_option( 'yith_wcan_enable_dropdown_price_filter' ),
					'wc_price_filter_dropdown_style'        => apply_filters( 'yith_wcan_dropdown_type', yith_wcan_get_option( 'yith_wcan_dropdown_style' ) ),
					'wc_price_filter_dropdown_widget_class' => yith_wcan_get_option( 'yith_wcan_ajax_widget_title_class', 'h3.widget-title' ),
					'widget_wrapper_class'                  => yith_wcan_get_option( 'yith_wcan_ajax_widget_wrapper_class', '.widget' ),
					'price_filter_dropdown_class'           => apply_filters( 'yith_wcan_dropdown_class', 'widget-dropdown' ),
					'ajax_pagination_enabled'               => yith_wcan_get_option( 'yith_wcan_enable_ajax_shop_pagination', 'no' ),
					'pagination_anchor'                     => yith_wcan_get_option( 'yith_wcan_ajax_shop_pagination', 'nav.woocommerce-pagination' ) . ' ' . yith_wcan_get_option( 'yith_wcan_ajax_shop_pagination_anchor_class', 'a.page-numbers' ),
					'force_widget_init'                     => apply_filters( 'yith_wcan_force_widget_init', false ),
				);

				wp_enqueue_script( 'yith_wcan_frontend-premium', YITH_WCAN_URL . 'assets/js/yith-wcan-frontend-premium' . $suffix . '.js', array( 'jquery' ), YITH_WCAN_VERSION, true );
				wp_localize_script( 'yith-wcan-script', 'yith_wcan_frontend', array( 'loader_url' => yith_wcan_get_option( 'yith_wcan_ajax_loader', $loader_url ) ) );
				wp_localize_script( 'yith_wcan_frontend-premium', 'yith_wcan_frontend_premium', $options );
			}
		}

		/**
		 * Add custom meta to filtered page
		 *
		 * @return void
		 */
		public function add_meta() {
			// add meta for parent class.
			parent::add_meta();

			// add meta for session.
			$session_share_url = $this->_query->get_current_session_share_url();

			if ( ! $session_share_url ) {
				return;
			}

			?>
			<meta name="yith_wcan:sharing_url" content="<?php echo esc_attr( $session_share_url ); ?>">
			<?php
		}

		/**
		 * Returns an array of parameters to use to localize shortcodes script
		 *
		 * @return array Array of parameters.
		 */
		protected function _get_shortcodes_localize() {
			return apply_filters(
				'yith_wcan_shortcodes_script_args',
				array(
					'query_param'           => YITH_WCAN_Query()->get_query_param(),
					'content'               => apply_filters( 'yith_wcan_content_selector', '#content' ),
					'change_browser_url'    => in_array( yith_wcan_get_option( 'yith_wcan_change_browser_url', 'yes' ), array( 'yes', 'custom' ) ),
					'instant_filters'       => 'yes' === yith_wcan_get_option( 'yith_wcan_instant_filters', 'yes' ),
					'ajax_filters'          => 'yes' === yith_wcan_get_option( 'yith_wcan_ajax_filters', 'yes' ),
					'show_clear_filter'     => 'yes' === yith_wcan_get_option( 'yith_wcan_show_clear_filter', 'no' ),
					'modal_on_mobile'       => 'yes' === yith_wcan_get_option( 'yith_wcan_modal_on_mobile', 'yes' ),
					'scroll_top'            => 'yes' === yith_wcan_get_option( 'yith_wcan_scroll_top', 'no' ),
					'scroll_target'         => false,
					'session_param'         => YITH_WCAN_Session_Factory::get_session_query_param(),
					'is_shop_on_front'      => is_shop(),
					'shop_url'              => trailingslashit( yit_get_woocommerce_layered_nav_link() ),
					'terms_per_page'        => apply_filters( 'yith_wcan_dropdown_terms_per_page', 10 ),
					'loader'                => 'custom' === yith_wcan_get_option( 'yith_wcan_ajax_loader_style', 'default' ) ? yith_wcan_get_option( 'yith_wcan_ajax_loader_custom_icon', '' ) : false,
					'mobile_media_query'    => 991,
					'show_current_children' => false,
					'currency_format'       => apply_filters(
						'yith_wcan_shortcodes_script_currency_format',
						array(
							'symbol'    => get_woocommerce_currency_symbol(),
							'decimal'   => esc_attr( wc_get_price_decimal_separator() ),
							'thousand'  => esc_attr( wc_get_price_thousand_separator() ),
							'precision' => wc_get_price_decimals(),
							'format'    => esc_attr( str_replace( array( '%1$s', '%2$s' ), array( '%s', '%v' ), get_woocommerce_price_format() ) ),
						)
					),
					'labels'                => apply_filters(
						'yith_wcan_shortcodes_script_labels',
						array(
							'empty_option' => _x( 'All', '[FRONTEND] "All" label shown when no term is selected', 'yith-woocommerce-ajax-navigation' ),
							'search_placeholder' => _x( 'Search...', '[FRONTEND] Search placeholder shown in terms dropdown', 'yith-woocommerce-ajax-navigation' ),
							'no_items' => _x( 'No item found', '[FRONTEND] Empty items list in the dropdown', 'yith-woocommerce-ajax-navigation' ),
							// translators: 1. Number of items to show.
							'show_more' => _x( 'Show %d more', '[FRONTEND] Show more link on terms dropdown', 'yith-woocommerce-ajax-navigation' ),
							'close' => _x( 'Close', '[FRONTEND] Alt text for modal close button on mobile', 'yith-woocommerce-ajax-navigation' ),
							'show_results' => _x( 'Show results', '[FRONTEND] Label for filter button, on mobile modal', 'yith-woocommerce-ajax-navigation' ),
							'clear_selection' => _x( 'Clear', '[FRONTEND] Label for clear selection link, that appears above filter after selection', 'yith-woocommerce-ajax-navigation' ),
							'clear_all_selections' => _x( 'Clear All', '[FRONTEND] Label for clear selection link, that appears above filter after selection', 'yith-woocommerce-ajax-navigation' ),
						)
					),
				)
			);
		}

		/* === TEMPLATE METHODS === */

		/**
		 * Print preset title template
		 *
		 * @param YITH_WCAN_Preset|bool $preset Current preset, when applicable; false otherwise.
		 *
		 * @return void
		 */
		public function filters_title( $preset = false ) {
			$title = yith_wcan_get_option( 'yith_wcan_filters_title', '' );

			/**
			 * Print title template when:
			 * 1. Admin set a title
			 * 2. Filters will be shown as modal on mobile (title will be shown on mobile only, default will apply if no filter is configured).
			 */
			if ( empty( $title ) && 'yes' !== yith_wcan_get_option( 'yith_wcan_modal_on_mobile' ) ) {
				return;
			}

			$additional_classes_array = array();

			// apply default title when required.
			if ( empty( $title ) ) {
				$title = apply_filters( 'yith_wcan_default_modal_title', _x( 'Filter products', '[FRONTEND] Default modal title - mobile only', 'yith-woocommerce-ajax-navigation' ) );
				$additional_classes_array[] = 'mobile-only';
			}

			$title_tag = apply_filters( 'yith_wcan_preset_title_tag', 'h3' );
			$additional_classes = implode( ' ', apply_filters( 'yith_wcan_preset_title_classes', $additional_classes_array, $this ) );

			echo wp_kses_post( sprintf( '<%1$s class="%3$s">%2$s</%1$s>', esc_html( $title_tag ), esc_html( $title ), esc_attr( $additional_classes ) ) );
		}

		/**
		 * Print Apply Filters button template
		 *
		 * @param YITH_WCAN_Preset|bool $preset Current preset, when applicable; false otherwise.
		 *
		 * @return void
		 */
		public function apply_filters_button( $preset = false ) {
			$instant_filters = 'yes' === yith_wcan_get_option( 'yith_wcan_instant_filters', 'yes' );

			if ( $instant_filters ) {
				return;
			}

			yith_wcan_get_template( 'filters/global/apply-filters.php', compact( 'preset' ) );
		}

		/**
		 * Hooks callback that will print list fo active filters
		 *
		 * @return void
		 */
		public function add_active_filters_list() {
			$show_active_filters = 'yes' === yith_wcan_get_option( 'yith_wcan_show_active_labels', 'yes' );
			$active_filters_position = yith_wcan_get_option( 'yith_wcan_active_labels_position', 'before_filters' );

			if ( ! $show_active_filters ) {
				return;
			}

			switch ( $active_filters_position ) {
				case 'before_filters':
					add_action( 'yith_wcan_before_preset_filters', array( $this, 'active_filters_list' ) );
					break;
				case 'after_filters':
					add_action( 'yith_wcan_after_preset_filters', array( $this, 'active_filters_list' ) );
					break;
				case 'before_products':
					$locations = $this->get_before_product_locations();

					if ( ! $locations ) {
						return;
					}

					foreach ( $locations as $location ) {
						add_action( $location['hook'], array( $this, 'active_filters_list' ), $location['priority'] );
					}
					break;
			}
		}

		/**
		 * Print list of active filters
		 *
		 * @param YITH_WCAN_Preset|bool $preset Current preset, when applicable; false otherwise.
		 *
		 * @return void
		 */
		public function active_filters_list( $preset = false ) {
			$active_filters = $this->_query->get_active_filters( 'view' );
			$show_titles    = 'yes' === yith_wcan_get_option( 'yith_wcan_active_labels_with_titles', 'yes' );
			$labels_heading = apply_filters( 'yith_wcan_active_filters_title', _x( 'Active filters', '[FRONTEND] Active filters title', 'yith-woocommerce-ajax-navigation' ) );

			yith_wcan_get_template( 'filters/global/active-filters.php', compact( 'preset', 'active_filters', 'show_titles', 'labels_heading' ) );
		}

		/**
		 * Adds Mobile Modal Opener button, before product sections when possible
		 *
		 * @return void
		 */
		public function add_mobile_modal_opener() {
			$modal_on_mobile = 'yes' === yith_wcan_get_option( 'yith_wcan_modal_on_mobile', 'no' );

			if ( ! $modal_on_mobile ) {
				return;
			}

			$locations = $this->get_before_product_locations( -2 );

			if ( ! $locations ) {
				return;
			}

			foreach ( $locations as $location ) {
				add_action( $location['hook'], array( $this, 'mobile_modal_opener' ), $location['priority'] );
			}
		}

		/**
		 * Print Mobile Modal Opener button
		 *
		 * @param YITH_WCAN_Preset|bool $preset Current preset, when applicable; false otherwise.
		 *
		 * @return void
		 */
		public function mobile_modal_opener( $preset = false ) {
			$preset = $preset instanceof YITH_WCAN_Preset ? $preset : false;
			$label = apply_filters( 'yith_wcan_mobile_modal_opener_label', _x( 'Filters', '[FRONTEND] Label for the Filters button on mobile', 'yith-woocommerce-ajax-navigation' ) );

			yith_wcan_get_template( 'filters/global/mobile-filters.php', compact( 'label', 'preset' ) );
		}

		/**
		 * Enqueue Script for Widget Dropdown
		 *
		 * @return void
		 * @author Andrea Grillo <andrea.grillo@yithemes.com
		 * @since  2.0
		 */
		public function add_dropdown_styles() {
			// Dropdown Options.
			$widget_title   = yith_wcan_get_option( 'yith_wcan_ajax_widget_title_class', 'h3.widget-title' );
			$widget_wrapper = yith_wcan_get_option( 'yith_wcan_ajax_widget_wrapper_class', '.widget' );
			$background_url = YITH_WCAN_URL . 'assets/images/select-arrow.png';

			$css = "{$widget_wrapper} {$widget_title}.with-dropdown {position: relative; cursor: pointer;}
                    {$widget_wrapper} {$widget_title}.with-dropdown .widget-dropdown { border-width: 0; width: 22px; height: 22px; background: url({$background_url}) top 0px right no-repeat; background-size: 95% !important; position: absolute; top: 0; right: 0;}
                    {$widget_wrapper} {$widget_title}.with-dropdown.open .widget-dropdown {background-position: bottom 15px right;}";

			wp_add_inline_style( 'yith-wcan-frontend', $css );
		}

		/**
		 * Add a body class(es)
		 *
		 * @param string $classes Body classes added by the plugin.
		 *
		 * @return string Filtered list of classes added by the plugin to the body.
		 * @since  1.0
		 * @author Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function premium_body_class( $classes ) {
			return 'yith-wcan-pro';
		}
	}
}
